unit IWExtCtrls;
{PUBDIST}

interface

uses
  {$IFDEF VSNET}
  System.ComponentModel, System.Drawing, System.IO,
  IWNetBaseControl,
  IWNetClasses,
  System.Drawing.Design, System.Drawing.Imaging,
  {$ELSE}
  Classes,
    {$IFDEF Linux}
    IWCLXComponent, IWCLXClasses,
    {$ELSE}
    IWVCLComponent, IWVCLClasses,
    {$ENDIF}
  {$ENDIF}
  {$IFDEF Linux}QGraphics, {$ELSE}Graphics, {$ENDIF}
  SysUtils,
  IWBaseHTMLControl, IWControl, IWTypes, IWFileReference, IWHTMLTag,
  IWScriptEvents, IWGlobal, IWColor, IWRenderContext, IWJpegOptions,
  IWContainerBorderOptions, IWBaseInterfaces, IWHTML40Interfaces,
  IWBaseHTMLComponent, IWBaseRenderContext;

type
  TIWCustomRadioGroup = class(TIWCustomControl, IIWInputControl, IIWSubmitControl, IIWTabOrder)
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  protected
    FSubmitParam : String;
    FItemIndex: Integer;
    FItems: TIWStringList;
    FLayout: TIWRadioGroupLayout;
    //
    procedure SetItemIndex(AIndex: Integer);
    procedure OnItemsChange( ASender : TObject );
    procedure SetItems(AValue: TIWStringList);
    procedure SetLayout(const AValue: TIWRadioGroupLayout);
    procedure SetValue(const AValue: string); virtual;
    procedure HookEvents(AContext: TIWPageContext40; AScriptEvents: TIWScriptEvents); override;
    procedure Submit(const AValue: string); override;
    procedure GetInputControlNames(ANames: TStringList); override;
    function IsForThisControl(AName: string): boolean; override;

    procedure InitControl; override;
    procedure DoClick; override;
  public
    function GetSubmitParam : String;
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
    function RenderStyle(AContext: TIWBaseHTMLComponentContext): string; override;

    property ItemIndex: integer read FItemIndex write SetItemIndex;
    {$IFDEF VSNET}
    [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
    {$ENDIF}
    property Items: TIWStringList read FItems write SetItems;

    property Layout: TIWRadioGroupLayout read FLayout write SetLayout;
  published
    // {$IFNDEF VSNET}
    property OnClick; //: TNotifyEvent read GetOnClick write SetOnClick;
    // {$ENDIF}
    property Confirmation;
  end;

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWRadioGroup.bmp}
  TIWRadioGroup = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWRadioGroup), 'TIWRadioGroup.bmp')]
  {$ENDIF}
  TIWRadioGroup = class(TIWCustomRadioGroup)
  published
    property Editable;
    property Enabled;
    {$IFDEF CLR}
    property WebFont;
    {$ELSE}
    property Font;
    {$ENDIF}
    property FriendlyName;
    property ItemIndex;
    property Items;
    property Layout;
    property ScriptEvents;
    property TabOrder;
  end;

  TIWImageOnMouseDown = procedure(ASender: TObject; const AX: Integer; const AY: Integer) of object;
  TIWImageOutput = (ioJPEG, ioGIF);

  TIWCustomImage = class(TIWCustomControl, IIWSubmitControl)
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  protected
    FAltText: string;
    FBorderOptions: TIWBorderOptions;
    FOnMouseDown: TIWImageOnMouseDown;
    FOutputType: TIWImageOutput;
    FPicture: TIWPicture;
    FSubmitParam : String;
    FUseSize: Boolean;
    {$IFDEF VSNET}
    FImage: Image;
    {$ENDIF}
    //
    function CanAutoSize(var VNewWidth: Integer; var VNewHeight: Integer): Boolean;
     {$IFNDEF Linux} override; {$ENDIF}
    procedure CreatePicture;
    // Necessary because Borland made FOnClick protected
    // {$IFNDEF VSNET}
    function GetOnClick: {$IFDEF VSNET}EventHandler {$ELSE} TNotifyEvent{$ENDIF};
    procedure SetOnClick(const AValue: {$IFDEF VSNET}EventHandler {$ELSE} TNotifyEvent{$ENDIF});
    // {$ENDIF}
    function ImageHTML(AContext: TIWBaseHTMLComponentContext; const ASrc: string): TIWHTMLTag;
    procedure HookEvents(AContext: TIWPageContext40; AScriptEvents: TIWScriptEvents); override;
    procedure PictureChanged(ASender: TObject);
    procedure SetOnMouseDown(const AValue: TIWImageOnMouseDown);
    procedure SetPicture(AValue: TIWPicture);
    function GetPicture: TIWPicture;
    {$IFDEF VSNET}
    procedure ConvertImageToPicture;
    procedure SetImage(AValue: Image);
    {$ENDIF}
    procedure Submit(const AValue: string); override;
    procedure SetUseSize(const AValue: Boolean);
    procedure SetBorderOptions(AValue: TIWBorderOptions);

    procedure SetAutoSize(AValue: Boolean); override;
    function get_HasTabOrder: Boolean; override;

    procedure InitControl; override;
    {$IFDEF VSNET}
    function ShouldSerializeImage: Boolean;
    [Bindable(true)]
    property Image: Image read FImage write SetImage;
    {$ENDIF}
  public
    function GetSubmitParam : String;
    property Picture: TIWPicture read GetPicture write SetPicture;
    property OutputType: TIWImageOutput read FOutputType write FOutputType;
  published
    {$IFDEF VSNET} [DefaultValue('')] {$ENDIF}
    property AltText: string read FAltText write FAltText;
{$IFNDEF Linux}
    // Necessary as Delphi defaults to False and causes streaming problems
    property AutoSize default True;
{$ENDIF}
    {$IFDEF VSNET}[DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]{$ENDIF}
    property BorderOptions: TIWBorderOptions read FBorderOptions write SetBorderOptions;
    property Confirmation;
    property DoSubmitValidation;
    property ExtraTagParams;
    property Enabled;
    property ScriptEvents;
    property TabOrder;
    property UseSize: Boolean read FUseSize write SetUseSize;
    //
    // Events
    //
    // {$IFNDEF VSNET}
    property OnClick: {$IFDEF VSNET}EventHandler {$ELSE} TNotifyEvent{$ENDIF} read GetOnClick write SetOnClick;
    // {$ENDIF}
    property OnMouseDown: TIWImageOnMouseDown read FOnMouseDown write SetOnMouseDown;
  end;

  TIWDynamicImage = class(TIWCustomImage)
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  protected
    FJpegOptions: TIWJpegOptions;
    FTransparentColor:TIWColor;
    //
    procedure SetJpegOptions(const AValue: TIWJpegOptions);

    procedure InitControl; override;
  public
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
    property TransparentColor:TIWColor read FTransparentColor write FTransparentColor;
    property JpegOptions: TIWJpegOptions read FJpegOptions write SetJpegOptions;
  published
  end;

{$IFDEF CLR}
// TIWDynamicImageFile requires GIF support, and we only have that on DotNet

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWDynamicImageFile.bmp}
  TIWDynamicImageFile = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWDynamicImageFile), 'TIWDynamicImageFile.bmp')]
  {$ENDIF}
  TIWDynamicImageFile = class(TIWDynamicImage)
  protected
    FImageFile: TIWFileReference;
    procedure SetImageFile(const AValue: TIWFileReference);
    procedure InitControl; override;
    procedure Submit(const AValue: string); override;
  public
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
  published
    {$IFDEF VSNET}
    property Image;
    {$ELSE}
    property Picture;
    {$ENDIF}
    property FriendlyName;
    property TransparentColor;
    {$IFDEF VSNET}
    [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
    [Bindable(true)]
    {$ENDIF}
    property ImageFile: TIWFileReference read FImageFile write SetImageFile;
  end;
  
{$ENDIF}

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWImage.bmp}
  TIWImage = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWImage), 'TIWImage.bmp')]
  {$ENDIF}
  TIWImage = class(TIWDynamicImage)
  published
    {$IFDEF VSNET}
    property Image;
    {$ELSE}
    property Picture;
    {$ENDIF}
    property FriendlyName;
    property TransparentColor;
    property JpegOptions;
    property OutputType;
  end;

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWImageFile.bmp}
  TIWImageFile = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWImageFile), 'TIWImageFile.bmp')]
  {$ENDIF}
  TIWImageFile = class(TIWCustomImage)
  {$IFDEF CLR} strict {$ENDIF} protected
    procedure Dispose(ADispose: Boolean); override;
  protected
    FCacheable: Boolean;
    FImageFile: TIWFileReference;
    //
    procedure InitControl; override;
    procedure Loaded; override;
    procedure ReferenceChanged(ASender: TObject);
    procedure SetImageFile(const AValue: TIWFileReference);
  public
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
  published
    property Cacheable: Boolean read FCacheable write FCacheable;
    property FriendlyName;
    {$IFDEF VSNET}
    [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
    [Bindable(true)]
    {$ENDIF}
    property ImageFile: TIWFileReference read FImageFile write SetImageFile;
  end;

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWImageButton.bmp}
  TIWImageButton = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWImageButton), 'TIWImageButton.bmp')]
  {$ENDIF}
  TIWImageButton = class(TIWImageFile)
  protected
    FHotKey: string;
    procedure HookEvents(AContext: TIWPageContext40; AScriptEvents: TIWScriptEvents); override;    
  public
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
  published
    {$IFDEF VSNET}
    [DefaultValue('')]
    {$ENDIF}
    property HotKey: string read FHotkey write FHotKey;
  end;


  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWTimer.bmp}
  TIWTimer = class;
  [
    ToolboxItem(true),
    ToolboxBitmap(typeof(TIWTimer), 'TIWTimer.bmp'),
    ToolboxItemFilter('TIWAppForm'),
    TIWToolPalette('Intraweb Controls')
  ]
  {$ENDIF}
  TIWTimer = class(TIWBaseHTMLComponent, IIWHTML40Component, IIWSubmitControl)
  protected
    FSubmitParam: string;
    FEnabled: Boolean;
    FInterval: Integer;
    FOnTimer: TNotifyEvent;
    //
    procedure Submit(const AValue: string); virtual;
    function GetSubmitProc: TSubmitProc;
    function get_ControlEncode: TIWControlEncode;
    procedure SetControlEncode(const AValue: TIWControlEncode);
    function ComponentContextClass: TIWBaseComponentContextClass; override;
    procedure InitControl; override;
    function get_DoRefreshControl: Boolean;
    procedure set_DoRefreshControl(AValue: Boolean);
    function RenderCSSClass(AComponentContext: TIWBaseHTMLComponentContext): string; virtual;
    function RenderStyle(AComponentContext: TIWBaseHTMLComponentContext): string; virtual;
    procedure RenderScripts(AComponentContext: TIWBaseHTMLComponentContext); virtual;
    function WantsSubmitValidation: Boolean;
    //
    property ControlEncode: TIWControlEncode read get_ControlEncode write SetControlEncode;
  public
    function GetSubmitParam : String;
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
  published
    property Enabled: Boolean read FEnabled write FEnabled;
    property Interval: Integer read FInterval write FInterval;
    property OnTimer: TNotifyEvent read FOnTimer write FOnTimer;
  end;

implementation

uses
  {$IFDEF VSNET}
  Classes, IWNetComponent,
  {$ELSE}
  {$IFDEF Linux}QControls,{$ELSE}Controls,{$ENDIF}
  {$ENDIF}
  IWGIF, IWServerControllerBase,
  SWSystem, SWStrings;

{ TIWRadioGroup }

procedure TIWCustomRadioGroup.InitControl;
begin
  inherited;
  FItems := TIWStringList.Create;
  TIWStringList(FItems).OnChange := OnItemsChange;  
  FNeedsFormTag := True;
  Width := 89;
  Height := 112;
end;

procedure TIWCustomRadioGroup.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FItems);
  inherited;
end;

procedure TIWCustomRadioGroup.Submit(const AValue: string);
begin
  FSubmitParam := AValue;
 { DoClick; { Call of DoClick is commented to prevent it to happen on every click
              because Delphi users are accostomed that OnClick is fire only when
              the value is changed. 
              Ivan }
end;

procedure TIWCustomRadioGroup.HookEvents(AContext: TIWPageContext40; AScriptEvents: TIWScriptEvents);
begin
  inherited HookEvents(AContext, AScriptEvents);
  if Editable then begin
    AScriptEvents.HookEvent('OnClick', iif(HasOnClick, SubmitHandler));
  end;
end;

function TIWCustomRadioGroup.GetSubmitParam: String;
begin
  Result := FSubmitParam;
end;

function TIWCustomRadioGroup.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
var
  i: Integer;
begin
  // To identify them as a group, HTML requires they all have the same name
  Result := TIWHTMLTag.CreateTag('DIV'); try
    for i := 0 to Items.Count - 1 do begin
      with Result.Contents.AddTag('INPUT') do begin
        AddStringParam('TYPE', 'RADIO');
        Add(IIF(ItemIndex = i, ' CHECKED'));
        AddStringParam('NAME', HTMLName + '_INPUT');
        AddStringParam('id', HTMLName + '_INPUT_'+IntToStr(i+1));
        AddStringParam('VALUE', IntToStr(i));
        Add(IIF((not Enabled) or (not Editable) , 'DISABLED'));
      end;
      with Result.Contents.AddTag('SPAN') do begin
        if Enabled and Editable then begin
          AddStringParam('OnClick', 'FindElem('''+HTMLName + '_INPUT_'+IntToStr(i+1)+ ''').checked = true;');
        end;
        Contents.AddText(Items.Strings[i]);
      end;
      Result.Contents.AddText(iif(Layout = glVertical, '<BR>', ' '));
    end;
  except FreeAndNil(Result); raise; end;
end;

function TIWCustomRadioGroup.RenderStyle(AContext: TIWBaseHTMLComponentContext): string;
begin
  Result := inherited RenderStyle(AContext);
  Result := Result + iif(FLayout = glHorizontal, ' white-space:nowrap;', '');
end;

procedure TIWCustomRadioGroup.OnItemsChange( ASender : TObject );
begin
  DoRefreshControl := True;
  Invalidate;
end;

procedure TIWCustomRadioGroup.SetItemIndex(AIndex: Integer);
begin
  if AIndex <> FItemIndex then begin
    FItemIndex := AIndex;
  DoRefreshControl := True;
  end;
end;

procedure TIWCustomRadioGroup.SetItems(AValue: TIWStringList);
begin
  FItems.Assign(AValue);
  Invalidate;
end;

procedure TIWCustomRadioGroup.SetLayout(const AValue: TIWRadioGroupLayout);
begin
  FLayout := AValue;
  Invalidate;
end;

procedure TIWCustomRadioGroup.DoClick;
begin
  { Inherited method is not called here because it will cause OnClick
    to be fired. However in this case OnClick will be fired always when
    it CheckBox is clicked even if the value is not changed (HTML feature)
    but Delphi users are used that IWCustomRadioGroup's OnClick event means that the value
    is changed. In TIWCustomRadioGroup, OnClick is called in SetValue proc.
    Ivan.
    }
//  inherited;
end;

procedure TIWCustomRadioGroup.SetValue(const AValue: string);
var
  LNewIndex : Integer;
begin
  if RequiresUpdateNotification(Parent) then begin
    UpdateNotifiedInterface(Parent).NotifyUpdate(Self,AValue);
  end;
  LNewIndex := StrToIntDef(AValue, -1);
  if ItemIndex <> LNewIndex then begin
    ItemIndex := LNewIndex;
    Invalidate;
    inherited DoClick;
  end; 
end; 

{ TIWCustomImage }

function TIWCustomImage.get_HasTabOrder: Boolean;
begin
  Result := Assigned(OnClick);
end;

function TIWCustomImage.GetSubmitParam: String;
begin
  Result := FSubmitParam;
end;

procedure TIWCustomImage.SetAutoSize(AValue: Boolean);
begin
  inherited;

  if AutoSize then begin
    UseSize := false;
    if Assigned(FPicture) then begin
      PictureChanged(nil);
    end;
  end;
end;

function TIWCustomImage.CanAutoSize(var VNewWidth, VNewHeight: Integer): Boolean;
begin
  Result := False;
  // FPicture is nil during initial create at design time for TIWImageFile (inherited Create)
  // Seems wierd to check AutoSize esp during streaming, but its required otehrwise during
  // AutoSize = False and streaming at runtime we get witdh and height of 0.
  if (FPicture <> nil) and (IsDesignMode = False) and (AutoSize = True) then begin
    if (FPicture.Width > 0) and (FPicture.Height > 0) then begin
      Result := True;
      if Align in [alNone, alLeft, alRight] then begin
        VNewWidth := FPicture.Width;
      end;
      if Align in [alNone, alTop, alBottom] then begin
        VNewHeight := FPicture.Height;
      end;
    end;
  end;
end;

procedure TIWCustomImage.InitControl; 
begin
  inherited;
  AutoSize := True;
  Width := 89;
  Height := 112;
  FBorderOptions := IWContainerBorderOptions.TIWBorderOptions.Create;
  FBorderOptions.Control := Self;
  FAltText := '';
end;

procedure TIWCustomImage.CreatePicture;
begin
  FPicture := TIWPicture.Create;
  FPicture.OnChange := PictureChanged;
end;

procedure TIWCustomImage.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FPicture);
  FreeAndNil(FBorderOptions);
  inherited;
end;

// {$IFNDEF VSNET}
function TIWCustomImage.GetOnClick:{$IFDEF VSNET}EventHandler {$ELSE} TNotifyEvent{$ENDIF};
begin
  Result := (inherited OnClick);
end;

procedure TIWCustomImage.SetOnClick(const AValue: {$IFDEF VSNET}EventHandler {$ELSE} TNotifyEvent{$ENDIF});
begin
  FOnMouseDown := nil;
  inherited OnClick := AValue;
end;
//{$ENDIF}

function TIWCustomImage.ImageHTML(AContext: TIWBaseHTMLComponentContext; const ASrc: string): TIWHTMLTag;
var
  LTag: TIWHTMLTag;
begin
  Result := TIWHTMLTag.CreateTag('IMG'); try
    Result.AddStringParam('SRC', ASrc);
    Result.AddStringParam('NAME', HTMLName);
    Result.AddIntegerParam('BORDER', BorderOptions.Width);
    if ToTColor(BorderOptions.Color) <> clNone then begin
      Result.Add('style={border-color: ' + ColorToRGBString(BorderOptions.Color) + ';}');
    end;
    Result.AddStringParam('ALT', AltText);
    if UseSize then begin
      Result.AddIntegerParam('WIDTH', Width);
      Result.AddIntegerParam('HEIGHT', Height);
    end;

    if Enabled then begin
      if Assigned(OnMouseDown) then begin
        if AContext.Browser in [brNetscape6, brNetscape7, brSafari] then begin
          LTag := TIWHTMLTag.CreateTag('A'); try
            LTag.AddStringParam('HREF', '#');
            LTag.AddStringParam('ID', HTMLName + '_IMAGE');
            LTag.AddStringParam('OnMouseOver', 'return ImageSetEvent(this,''' + HTMLName + ''''
              + ',' + iif(DoSubmitValidation, 'true', 'false')
              + ',' + '''' + TextToJSStringLiteral(Confirmation) + ''''
              + ');"');
            LTag.Contents.AddTagAsObject(Result, true);
          except FreeAndNil(LTag); raise; end;
        end;
      end;
      // OnMouseDown and OnClick are mutually exclusive, no need to check
      // TODO: Fix for .NET
      if HasOnClick then begin
        LTag := SubmitLink('', '', False,  AContext.Browser);
        LTag.Contents.AddTagAsObject(Result, true);
        //Result := LTag;
      end;
    end;
  except FreeAndNil(Result); raise; end;
end;

procedure TIWCustomImage.PictureChanged(ASender: TObject);
begin
  if not IsLoading then begin
    if AutoSize and (FPicture.Width > 0) and (FPicture.Height > 0) then begin
      SetBounds(Left, Top, FPicture.Width, FPicture.Height);
    end;
    Invalidate;
  end;
end;

procedure TIWCustomImage.SetOnMouseDown(const AValue: TIWImageOnMouseDown);
begin
  inherited OnClick := nil;
  FOnMouseDown := AValue;
end;

procedure TIWCustomImage.Submit(const AValue: string);
var
  s: string;
  LX: Integer;
  LY: Integer;
begin
  FSubmitParam := AValue;
  if Assigned(OnMouseDown) then begin
    s := AValue;
    LX := StrToInt(Fetch(s, ','));
    LY := StrToInt(s);
    OnMouseDown(Self, LX, LY);
  end else begin
    DoClick;
  end;
end;

procedure TIWCustomImage.SetUseSize(const AValue: Boolean);
begin
  set_RenderSize(AValue);
  FUseSize := AValue;
  if FUseSize then
    AutoSize := false;
end;

procedure TIWCustomImage.SetBorderOptions(AValue: TIWBorderOptions);
begin
  FBorderOptions.Assign(AValue);
end;

procedure TIWCustomImage.HookEvents(AContext: TIWPageContext40; AScriptEvents: TIWScriptEvents);
begin
  inherited;
  if Assigned(OnMouseDown) then begin
    if AContext.WebApplication.Browser in [brIE, brOpera] then begin
      AScriptEvents.HookEvent('OnMouseDown', 'return ImageClick(this.name,event,'
        + iif(DoSubmitValidation, 'true', 'false') + ', ''' + TextToJSStringLiteral(Confirmation) + ''');');
    end;
  end;
end;

{$IFDEF VSNET}
function TIWCustomImage.ShouldSerializeImage: Boolean;
begin
  result := Assigned(FImage); 
end;

procedure TIWCustomImage.ConvertImageToPicture;
Var
  ms: MemoryStream;
  ms1: TCLRStreamWrapper;
begin
    if Assigned(FImage) then begin
    ms := MemoryStream.Create; try
      FImage.Save(ms, ImageFormat.Bmp);
      ms1 := TCLRStreamWrapper.Create(ms); try
        Picture.Graphic := TBitmap.Create;
        ms1.Position := 0;
        Picture.Graphic.LoadFromStream(ms1);
      finally
        ms1.Free;
      end;
    finally
      ms.Free;
    end;
  end else begin
    Picture.Graphic := nil;
  end;
end;

procedure TIWCustomImage.SetImage(AValue: Image);
begin
  FImage := AValue;
  ConvertImageToPicture;
end;
{$ENDIF}

function TIWCustomImage.GetPicture: TIWPicture;
begin
  if {IsDesignMode and }not Assigned(FPicture) then begin
    CreatePicture;
  end;
  result := FPicture;
end;

procedure TIWCustomImage.SetPicture(AValue: TIWPicture);
begin
  Picture.Assign(AValue);
  // No need for Invalidate, Picture.OnChange will catch it
end;

{ TIWImage }

procedure TIWDynamicImage.InitControl;
begin
  inherited;
  FJpegOptions := IWJpegOptions.TIWJpegOptions.Create;
  {$IFDEF VSNET}
  FTransparentColor := System.Drawing.Color.Empty;
  {$ELSE}
  FTransparentColor := clNone;
  {$ENDIF}
end;

procedure TIWDynamicImage.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FJpegOptions);
  inherited;
end;

function TIWDynamicImage.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
var
  LExt: string;
  LPathname: string;
begin
  // Is nil if no picture. Ex: Drop a TIWImage on form, dont load an image, run.
  if Picture.Graphic <> nil then begin
    if not FPicture.Graphic.Empty then begin
      {$IFDEF VSNET} ConvertImageToPicture; {$ENDIF}
      case OutputType of
        ioJPEG: LExt := 'jpg';
        ioGIF: LExt := 'gif';
        else raise Exception.Create('Unknown Image Output Type');
      end;
      LPathname := TIWServerControllerBase.NewCacheFile(LExt, True);
      case OutputType of
        ioJPEG: ToJpegFile(FPicture.Graphic, LPathname, FJpegOptions);
        ioGIF: ToGifFile(FPicture.Graphic, LPathname, TransparentColor)
      end;
      Result := ImageHTML(AContext, GServerController.UserCacheURL + ExtractFilename(LPathname));
    end else begin
      Result := TIWHTMLTag.CreateTag('span');
    end;
  end else begin
    Result := TIWHTMLTag.CreateTag('span');
  end;
end;

procedure TIWDynamicImage.SetJpegOptions(const AValue: TIWJpegOptions);
begin
  FJpegOptions.Assign(AValue);
end;

{ TIWImageFile }

procedure TIWImageFile.InitControl;
begin
  inherited;
  FCacheable := True;
  FImageFile := TIWFileReference.Create;
  FImageFile.OnChange := ReferenceChanged;
end;

procedure TIWImageFile.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FImageFile);
  inherited;
end;

procedure TIWImageFile.ReferenceChanged(ASender: TObject);
var
  LBitmap : TBitmap;
begin
  if IsDesignMode then
  begin
    if FileExists(ImageFile.Filename) then
    begin
      try
        Picture.LoadFromFile(ImageFile.Filename);
      except
        on EInvalidGraphic do
        begin
          LBitmap := TBitmap.Create;
          try
            LBitmap.Width := Width;
            LBitmap.Height := Height;
            LBitmap.Canvas.Font.Color := clRed;
            LBitmap.Canvas.TextOut( 5, 5, 'Please see the FAQ about' );
            LBitmap.Canvas.TextOut( 5, 20, 'enabling GIF support' );
            LBitmap.Canvas.TextOut( 5, 35, 'at design time' );
            Picture.Bitmap := LBitmap;
          finally
            FreeAndNil( LBitmap );
          end;
        end;
      end;
    end
    else
    begin
      Picture.Graphic := nil;
    end;
    Invalidate;
  end
  else
  begin
    if AutoSize and (FPicture <> nil) then
    begin
      if (FPicture.Graphic.Width > 0) and (FPicture.Graphic.Height > 0) then
      begin
        Width := FPicture.Graphic.Width;
        Height := FPicture.Graphic.Height;
      end;
    end;
    DoRefreshControl := TRUE;
  end;
end;

function TIWImageFile.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
begin
  Cacheable := (Pos('FILESNC', UpperCase(FImageFile.URL)) = 0) and (Pos('FILESNC', UpperCase(FImageFile.Filename)) = 0);
  Result := ImageHTML(AContext, ImageFile.Location(iif(Cacheable, GServerController.FilesURL
   , GServerController.FilesNCURL)));
end;

procedure TIWImageFile.Loaded;
begin
  inherited;

  if IsDesignMode then begin
    if AutoSize and (Picture.Width > 0) and (Picture.Height > 0) then begin
      SetBounds(Left, Top, Picture.Width, Picture.Height);
    end;
    Invalidate;
  end;
end;

procedure TIWImageFile.SetImageFile(const AValue: TIWFileReference);
begin
  // Assign calls DoChange
  FImageFile.Assign(AValue);
end;

{ TIWDynamicImageFile }

{$IFDEF CLR}

procedure TIWDynamicImageFile.InitControl;
begin
  inherited;
  FImageFile := TIWFileReference.Create;
  OutputType := ioGIF;
  AutoSize := false;
  {$IFDEF VSNET}
  Image := System.Drawing.Bitmap.Create(Width, Height, PixelFormat.Format24bppRgb);
  {$ELSE}
  Picture.Graphic := TBitmap.Create;
  TBitmap(Picture.Graphic).Width := Width;
  TBitmap(Picture.Graphic).Height := Height;
  {$ENDIF}
//  ImageFormat := ImageFormat.Gif;
  {$IFDEF VSNET}
  FTransparentColor := System.Drawing.Color.Black;
  {$ELSE}
  FTransparentColor := clWhite;
  {$ENDIF}
end;

procedure TIWDynamicImageFile.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
begin
  inherited SetBounds(ALeft, ATop, AWidth, AHeight);
  {$IFDEF VSNET}
  Image := System.Drawing.Bitmap.Create(Image, Width, Height);
  {$ELSE}
  if Picture.Graphic is TBitmap then begin
    TBitmap(Picture.Graphic).Width := Width;
    TBitmap(Picture.Graphic).Height := Height;
  end;
  {$ENDIF}
end;


procedure TIWDynamicImageFile.Submit(const AValue: string);
var
  s: string;
  LX: Integer;
  LY: Integer;
begin
  FSubmitParam := AValue;
  if Assigned(OnMouseDown) then begin
    s := AValue;
    LX := StrToInt(Fetch(s, ','));
    LY := StrToInt(s);
    OnMouseDown(Self, LX, LY);
  end else begin
    DoClick;
  end;
end;

function TIWDynamicImageFile.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
var
  LPathname: string;
  i:TIWHTMLTag;
begin
  Result := TIWHTMLTag.CreateTag('SPAN');
  Result.AddStringParam('NAME', HTMLName);
  Result.AddStringParam('STYLE','overflow:hidden;');

  // Is nil if no picture. Ex: Drop a TIWImage on form, dont load an image, run.
  if Picture.Graphic <> nil then begin
    if not FPicture.Graphic.Empty then begin
      {$IFDEF VSNET} ConvertImageToPicture; {$ENDIF}
      LPathname := TIWServerControllerBase.NewCacheFile('gif', True);
      ToGifFile(FPicture.Graphic, LPathname, TransparentColor);
      i:=ImageHTML(AContext, GServerController.UserCacheURL + ExtractFilename(LPathname));
      i.AddIntegerParam('WIDTH', Width);
      i.AddIntegerParam('HEIGHT', Height);
      i.AddStringParam('STYLE','position:relative;left:0;top:0;height:'+IntToStr(Height)+';width:'+IntToStr(Width)+';z-index:'+IntToStr(ZIndex+5)+';');
      Result.Contents.AddTagAsObject(i);
    end;
  end;

  with Result.Contents.AddTag('IMG') do begin
    AddStringParam('SRC', ImageFile.Location(GServerController.FilesURL));
    AddIntegerParam('BORDER', BorderOptions.Width);
//    AddStringParam('STYLE','position:relative;left:0;top:-'+inttostr(Height)+';height:'+IntToStr(Height)+';width:'+IntToStr(Width)+';z-index:'+IntToStr(ZIndex)+';');
    AddStringParam('STYLE','position:relative;left:0;top:-'+inttostr(Height)+';z-index:'+IntToStr(ZIndex)+';');
//    if toTColor(BorderOptions.Color) <> clNone then begin
//      Add('style=border-color: ' + ColorToRGBString(BorderOptions.Color));
//    end;
  end;

end;

procedure TIWDynamicImageFile.SetImageFile(const AValue: TIWFileReference);
begin
  FImageFile.Assign(AValue);
end;

{$ENDIF}

{ TIWTimer }

procedure TIWTimer.InitControl;
begin
  inherited;
  FEnabled := true;
  Interval := 1000;
end;

function TIWTimer.get_ControlEncode: TIWControlEncode;
begin
  result := ceNone;
end;

function TIWTimer.GetSubmitParam: String;
begin
  Result := FSubmitParam;
end;

function TIWTimer.GetSubmitProc: TSubmitProc;
begin
  result := Submit;
end;

function TIWTimer.ComponentContextClass: TIWBaseComponentContextClass;
begin
  result := TIWComponent40Context;
end;

function TIWTimer.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
begin
  result := nil;
end;

procedure TIWTimer.SetControlEncode(const AValue: TIWControlEncode);
begin
  // 
end;

procedure TIWTimer.Submit(const AValue: string);
begin
  FSubmitParam := AValue;
  if Assigned(OnTimer) then begin
    OnTimer(Self);
  end;
end;

function TIWTimer.get_DoRefreshControl: Boolean;
begin
  result := false;
end;

procedure TIWTimer.set_DoRefreshControl(AValue: Boolean);
begin
  // 
end;

function TIWTimer.RenderCSSClass(AComponentContext: TIWBaseHTMLComponentContext): string;
begin
  result := '';
end;

function TIWTimer.RenderStyle(AComponentContext: TIWBaseHTMLComponentContext): string;
begin
  result := '';
end;

procedure TIWTimer.RenderScripts(AComponentContext: TIWBaseHTMLComponentContext); 
var
  s : string;
  LContext : TIWComponent40Context;
begin
  LContext := TIWComponent40Context(AComponentContext);
  if Enabled and (Interval > 0) then begin
    // timer is enabled
    s := 'IWTop().' + HTMLName + '_ID=setTimeout(''IWTop().SubmitClickConfirm("' + HTMLName
        + '","", false,"")'',' + IntToStr(Interval) + ');' + EOL;

    if LContext.WebApplication.IsPartialUpdate then begin
      LContext.AddToUpdateInitProc( s );
    end else begin
      LContext.AddToInitProcDoLastCode( s );
    end;
  end else begin
    // not enabled
    if LContext.WebApplication.IsPartialUpdate then begin
      LContext.AddToUpdateInitProc(
        'if(IWTop().' + HTMLName + '_ID){clearInterval(IWTop().' + HTMLName + '_ID);};' + EOL );
    end else begin
      LContext.AddToInitProc('IWTop().' + HTMLName + '_ID=null;' + EOL );
    end;
  end;
end;

function TIWTimer.WantsSubmitValidation: Boolean;
begin
  result := false;
end;

procedure TIWCustomRadioGroup.GetInputControlNames(ANames: TStringList);
begin
  ANames.Add(HTMLName + '_INPUT')
end;

function TIWCustomRadioGroup.IsForThisControl(AName: string): boolean;
begin
  result := AName = HTMLName + '_INPUT';
end;

function TIWImageButton.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
var
  LSrc : String;
begin
  Cacheable := (Pos('FILESNC', UpperCase(FImageFile.URL)) = 0) and (Pos('FILESNC', UpperCase(FImageFile.Filename)) = 0);
  LSrc := ImageFile.Location(iif(Cacheable, GServerController.FilesURL, GServerController.FilesNCURL));

  Result := TIWHTMLTag.CreateTag('input'); try
    Result.AddStringParam('type','image');
    Result.AddStringParam('src',LSrc);
    Result.AddStringParam('name', HTMLName);
    if FHotKey <> '' then begin
      Result.AddStringParam('ACCESSKEY', FHotKey);
    end;
    if not Enabled then begin
      Result.Add('disabled');
    end;

  except FreeAndNil(Result); raise; end;
end;

procedure TIWImageButton.HookEvents(AContext: TIWPageContext40; AScriptEvents: TIWScriptEvents);
begin
  inherited HookEvents(AContext, AScriptEvents);
  AScriptEvents.HookEvent('OnClick', iif(hasOnClick, SubmitHandler('')));
end;

end.
